####################################################################
#############################   PETPA   ############################
####################################################################
# Loop over a series of given frequencies
# compute 2nd hyperpolarizabilities of p-NA in vacuum
# compute 2nd hyperpolarizabilities of p-NA in the PIM environment
# Calculate the TPA cross-sections for both cases
# Compare the ratio to obtain the PETPA enhancement factor
####################################################################


######################### Initializations ##########################
import os, sys
import numpy as np
from scm.plams import *
init()


############################ Filenames #############################
geometryFileQM   = 'pNA.xyz'
geometryFileDIM  = 'Ag2057.xyz'
parameterFileDIM = 'Ag_jc'


######################### Unit Conversion ##########################
# conversion a.u. to GM i.e. to 10E-50 * cm^4 * s / photon
au2GM    = 1.896831653271693 # = 5.2918e-9**4 * 2.418884326505E-17 * 1.0E50
alphaf   = 0.0072973525664
factorGM = au2GM * 4 * np.pi**3 * alphaf**2 / 15


############################## Models ##############################
# Load structure of the QM region
molQM = Molecule(geometryFileQM)

# Generate the DIM region, load and shift one cluster, then rotate its copy and merge both parts into a single molecule instance
dimRegionL = Molecule(geometryFileDIM)
dimRegionL.translate((.0, .0, -4.-2.81396650))
dimRegionU = dimRegionL.copy()
dimRegionU.rotate([-1., .0, .0, .0, 1., .0, .0, .0, -1])
dimRegion = dimRegionL + dimRegionU


########################## DFT-Settings #############################
# QM Settings
setQM = Settings()
setQM.input.ams.Task                   = 'SinglePoint'
setQM.input.adf.basis.type             = 'TZP'
setQM.input.adf.basis.core             = 'None'
setQM.input.adf.xc.model               = 'SAOP'
setQM.input.adf.symmetry               = 'nosym'
setQM.input.adf.allpoints              = ''
setQM.input.adf.numericalquality       = 'good'

# Just a single frequency of 1.71 eV:
omegas                                 = [ 1.71 ]
# Alternatively an entire spectrum:
#omegas                                = np.linspace(0.864, 2.48, num=20)
setQM.input.adf.aoresponse.scf         = 'iterations 50'
setQM.input.adf.aoresponse.gamma       = ''
setQM.input.adf.aoresponse.tpa         = ''
setQM.input.adf.aoresponse.ALDA        = ''
setQM.input.adf.aoresponse.lifetime    = '0.0037'

########################## PIM-Settings #############################
# DIM Settings, generates the blocks DIMQM and DIMPAR
setDIM = Settings()
setDIM.input.adf.DIMQM.PIM        = ''
setDIM.input.adf.DIMQM.ALGORITHM  = 'Direct'
setDIM.input.adf.DIMQM.NITER      = '2000'
setDIM.input.adf.DIMQM.FREQUENCY  = ''
setDIM.input.adf.DIMQM.LOCALFIELD = ''

setDIM.input.adf.DIMPAR['_1']     = 'Ag'
setDIM.input.adf.DIMPAR['_2']     = 'rad = 1.4445'
setDIM.input.adf.DIMPAR['_3']     = 'exp '+ os.path.join(os.path.dirname(os.path.abspath(__file__)), parameterFileDIM)
setDIM.input.adf.DIMPAR['_4']     = 'SUBEND'
setDIM.input.adf.DIMPAR['_5']     = 'XYZ'
setDIM.input.adf.DIMPAR['_6']     = len(dimRegion)
for iAt, atom in enumerate(dimRegion):
   setDIM.input.adf.DIMPAR[f'_{7+iAt}'] = atom.str()
setDIM.input.adf.DIMPAR[f'_{7+len(dimRegion)}'] = 'SUBEND'


### Loop over frequencies and calculate PETPA enhancement factors ###
spectrum = [('omega', 'sigma(TPA)', 'sigma(PETPA)', 'enhancement')]
for omega in omegas:
    setQM.input.adf.aoresponse.Frequencies = f"{omega} {omega} {-omega}"

    ################### Calculate TPA Cross Sections ####################
    # Setup and run TPA job without DIM environment
    jobTPA         = AMSJob(name=f'TPA_{omega}eV', molecule=molQM, settings=setQM)
    resultsTPA     = jobTPA.run()
    gammaImagTPA   = np.array([float(elem.split()[-1]) for elem in resultsTPA.grep_output(pattern=' gamma                             real              imaginary', options='-A81')[1:]]).reshape((3,3,3,3))
    sigmaTPA       = factorGM * Units.convert(omega,'eV','au')**2 * sum(gammaImagTPA[i][i][j][j] + gammaImagTPA[i][j][j][i] + gammaImagTPA[i][j][i][j] for i in range(3) for j in range(3))

    # Setup and run PETPA job with DIM environment
    jobPETPA       = AMSJob(name=f'PETPA_{omega}eV', molecule=molQM, settings=setQM+setDIM)
    resultsPETPA   = jobPETPA.run()
    gammaImagPETPA = np.array([float(elem.split()[-1]) for elem in resultsPETPA.grep_output(pattern=' gamma                             real              imaginary', options='-A81')[1:]]).reshape((3,3,3,3))
    sigmaPETPA     = factorGM * Units.convert(omega,'eV','au')**2 * sum(gammaImagPETPA[i][i][j][j] + gammaImagPETPA[i][j][j][i] + gammaImagPETPA[i][j][i][j] for i in range(3) for j in range(3))

    ######################## Gather Results #############################
    spectrum.append((omega, sigmaTPA, sigmaPETPA, sigmaPETPA/sigmaTPA))
    print('omega          = ',  omega, 'eV')
    print('gammaI(TPA)    =\n', gammaImagTPA)
    print('gammaI(PETPA)  =\n', gammaImagPETPA)
    print('sigma(TPA)     = ',  sigmaTPA, 'GM')
    print('sigma(PETPA)   = ',  sigmaPETPA, 'GM')
    sys.stdout.flush()


######################## Print Final Results ########################
for omega, sigmaTPA, sigmaPETPA, enhancement in spectrum:
   print(omega, sigmaTPA, sigmaPETPA, enhancement)

finish()
